/*
 * This file is part of WrapperLib
 * Copyright 2022 LukeGrahamLandry
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at https://mozilla.org/MPL/2.0/.
 */

package ca.lukegrahamlandry.lib.network.fabric;

import ca.lukegrahamlandry.lib.base.GenericHolder;
import ca.lukegrahamlandry.lib.base.event.IEventCallbacks;
import ca.lukegrahamlandry.lib.network.NetworkWrapper;
import net.fabricmc.fabric.api.client.networking.v1.ClientPlayNetworking;
import net.fabricmc.fabric.api.networking.v1.PacketByteBufs;
import net.fabricmc.fabric.api.networking.v1.PlayerLookup;
import net.fabricmc.fabric.api.networking.v1.ServerPlayNetworking;
import net.minecraft.class_1297;
import net.minecraft.class_2596;
import net.minecraft.class_3222;
import net.minecraft.server.MinecraftServer;

public class NetworkWrapperImpl implements IEventCallbacks {
    private static MinecraftServer SERVER;

    @Override
    public void onInit(){
        ServerPlayNetworking.registerGlobalReceiver(NetworkWrapper.ID, (server, player, handler, buf, responseSender) -> {
            GenericHolder<?> message = GenericHolder.decodeBytes(buf);
            server.execute(() -> NetworkWrapper.handleServerPacket(player, message));
        });
    }

    @Override
    public void onClientSetup() {
        ClientPlayNetworking.registerGlobalReceiver(NetworkWrapper.ID, (client, handler, buf, responseSender) -> {
            GenericHolder<?> message = GenericHolder.decodeBytes(buf);
            client.execute(() -> NetworkWrapper.handleClientPacket(message));
        });
    }

    @Override
    public void onServerStarting(MinecraftServer server) {
        SERVER = server;
    }

    @Override
    public void onServerStopped(MinecraftServer server) {
        SERVER = null;
    }

    public static <T> void sendToServer(T message){
        ClientPlayNetworking.send(NetworkWrapper.ID, new GenericHolder<>(message).encodeBytes(PacketByteBufs.create()));
    }

    public static <T> void sendToClient(class_3222 player, T message){
        ServerPlayNetworking.send(player, NetworkWrapper.ID, new GenericHolder<>(message).encodeBytes(PacketByteBufs.create()));
    }

    public static <T> void sendToAllClients(T message){
        SERVER.method_3760().method_14571().forEach((player) -> sendToClient(player, message));
    }

    public static <T> void sendToTrackingClients(class_1297 entity, T message){
        PlayerLookup.tracking(entity).forEach((p) -> sendToClient(p, message));
    }

    public static <T> class_2596<?> toVanillaPacket(T message, boolean isClientBound){
        if (isClientBound) return ServerPlayNetworking.createS2CPacket(NetworkWrapper.ID, new GenericHolder<>(message).encodeBytes(PacketByteBufs.create()));
        else return ClientPlayNetworking.createC2SPacket(NetworkWrapper.ID, new GenericHolder<>(message).encodeBytes(PacketByteBufs.create()));
    }
}
