/*
 * This file is part of WrapperLib
 * Copyright 2022 LukeGrahamLandry
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at https://mozilla.org/MPL/2.0/.
 */

package ca.lukegrahamlandry.lib.resources;

import ca.lukegrahamlandry.lib.base.Available;
import ca.lukegrahamlandry.lib.base.InternalUseOnly;
import ca.lukegrahamlandry.lib.base.json.JsonHelper;
import ca.lukegrahamlandry.lib.config.ConfigWrapper;
import ca.lukegrahamlandry.lib.helper.PlatformHelper;
import com.google.gson.Gson;
import com.google.gson.JsonElement;
import com.google.gson.JsonSyntaxException;
import com.google.gson.reflect.TypeToken;
import dev.architectury.injectables.annotations.ExpectPlatform;
import org.slf4j.Logger;

import java.io.BufferedReader;
import java.util.*;
import net.minecraft.class_2960;
import net.minecraft.class_3298;
import net.minecraft.class_3300;
import net.minecraft.class_3695;
import net.minecraft.class_4080;

public class ResourcesWrapper<T> extends class_4080<Map<class_2960, List<JsonElement>>> {
    /**
     * Load information from a data pack on the logical server.
     */
    public static <T> ResourcesWrapper<T> data(Class<T> clazz, String directory){
        return new ResourcesWrapper<>(TypeToken.get(clazz), directory, true);
    }

    /**
     * Load information from a resource pack on the logical client.
     */
    public static <T> ResourcesWrapper<T> assets(Class<T> clazz, String directory){
        return new ResourcesWrapper<>(TypeToken.get(clazz), directory, false);
    }

    public ResourcesWrapper<T> mergeWith(MergeRule<T> rule){
        this.mergeRule = rule;
        return this;
    }

    public ResourcesWrapper<T> onLoad(Runnable action){
        this.onLoadAction = action;
        return this;
    }

    public ResourcesWrapper<T> synced(){
        if (!this.isServerSide) throw new RuntimeException("ResourcesWrapper#synced may only be called for data packs, NOT resource packs.");
        if (!Available.NETWORK.get()) throw new RuntimeException("Called ResourcesWrapper#synced but WrapperLib Network module is missing.");

        this.shouldSync = true;
        return this;
    }

    public ResourcesWrapper<T> withGson(Gson gson){
        this.gson = gson;
        return this;
    }

    // API

    public Set<Map.Entry<class_2960, T>> entrySet(){
        return data.entrySet();
    }

    public T get(class_2960 id){
        return data.get(id);
    }

    // IMPL

    static List<ResourcesWrapper<?>> ALL = new ArrayList<>();
    TypeToken<T> valueType;
    public final String directory;
    public final boolean isServerSide;
    Map<class_2960, T> data;
    Logger logger;
    protected String suffix = ".json";
    private Gson gson = JsonHelper.get();
    boolean shouldSync = false;
    protected Runnable onLoadAction = () -> {};
    protected MergeRule<T> mergeRule = (resources) -> resources.get(resources.size() - 1);  // TODO: make sure I guessed the priority order right
    public ResourcesWrapper(TypeToken<T> valueType, String directory, boolean isServerSide){
        this.valueType = valueType;
        this.directory = directory;
        this.isServerSide = isServerSide;
        if (!this.isServerSide && Available.PLATFORM_HELPER.get() && PlatformHelper.isDedicatedServer()) return;
        ALL.add(this);
        registerResourceListener(this);
    }

    public Gson getGson(){
        return this.gson;
    }

    /**
     * For each filename in our directory of each pack we retrieve the list of files.
     * Each file comes from a different pack. They all get added to a list under their resource location.
     * This map with all the entries from every pack gets returned and later passed to PackWrapper#apply.
     */
    @Override
    protected Map<class_2960, List<JsonElement>> method_18789(class_3300 resourceManager, class_3695 profilerFiller) {
        Map<class_2960, List<JsonElement>> results = new HashMap<>();
        for (Map.Entry<class_2960, List<class_3298>> entry : resourceManager.method_41265(this.directory, file -> file.method_12832().endsWith(suffix)).entrySet()) {
            String actualPath = entry.getKey().method_12832().substring(this.directory.length() + 1, entry.getKey().method_12832().length() - suffix.length());
            class_2960 id = new class_2960(entry.getKey().method_12836(), actualPath);
            List<JsonElement> values = new ArrayList<>();
            for (class_3298 resource : entry.getValue()) {
                try (BufferedReader reader = resource.method_43039()) {
                    values.add(this.getGson().fromJson(reader, JsonElement.class));
                } catch (JsonSyntaxException e) {
                    this.logger.error("Failed to parse json for " + id + " from pack " + resource.method_14480());
                    e.printStackTrace();
                } catch (Exception e) {
                    this.logger.error("Failed to read " + id + " from pack " + resource.method_14480());
                    e.printStackTrace();
                }
            }
            if (!values.isEmpty()) results.put(id, values);
        }
        return results;
    }

    /**
     * For each resource location, all the json elements get parsed into our data type.
     * The resultant list is passed to the merge rule and its result is saved.
     */
    @Override
    protected void apply(Map<class_2960, List<JsonElement>> resources, class_3300 resourceManager, class_3695 profilerFiller) {
        this.data = new HashMap<>();
        for (Map.Entry<class_2960, List<JsonElement>> entry : resources.entrySet()){
            class_2960 id = entry.getKey();
            List<JsonElement> resourceStack = entry.getValue();
            List<T> values = new ArrayList<>();

            for (JsonElement json : resourceStack){
                values.add(this.getGson().fromJson(json, this.valueType.getType()));
            }

            T finalValue = this.mergeRule.merge(values);
            this.data.put(id, finalValue);
        }
        this.onLoadAction.run();
        if (this.shouldSync) new DataPackSyncMessage(this).sendToAllClients();
    }

    @InternalUseOnly
    void set(Object value) {
        this.data = (Map<class_2960, T>) value;
    }

    /**
     * Platform specific registration of a resource reload listener.
     */
    @ExpectPlatform
    public static void registerResourceListener(ResourcesWrapper<?> wrapper){
        throw new AssertionError();
    }
}
